// ===== STARBUCKS HACKER GAME - AUDIO MANAGER =====

const AudioManager = {
    sounds: {},
    bgMusic: null,
    isMuted: false,
    
    // Initialize audio context and sounds
    init() {
        // Create audio context for generating sounds
        try {
            this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
        } catch (e) {
            console.log('Web Audio API not supported');
            return;
        }
    },
    
    // Generate a beep sound
    generateBeep(frequency = 440, duration = 0.1, type = 'square') {
        if (!this.audioContext) return;
        
        const oscillator = this.audioContext.createOscillator();
        const gainNode = this.audioContext.createGain();
        
        oscillator.connect(gainNode);
        gainNode.connect(this.audioContext.destination);
        
        oscillator.frequency.value = frequency;
        oscillator.type = type;
        
        gainNode.gain.setValueAtTime(0.3, this.audioContext.currentTime);
        gainNode.gain.exponentialRampToValueAtTime(0.01, this.audioContext.currentTime + duration);
        
        oscillator.start(this.audioContext.currentTime);
        oscillator.stop(this.audioContext.currentTime + duration);
    },
    
    // Play typing sound
    playTyping() {
        if (this.isMuted) return;
        this.generateBeep(800 + Math.random() * 400, 0.02, 'square');
    },
    
    // Play success sound
    playSuccess() {
        if (this.isMuted) return;
        
        // Ascending notes
        setTimeout(() => this.generateBeep(523, 0.1, 'sine'), 0);
        setTimeout(() => this.generateBeep(659, 0.1, 'sine'), 100);
        setTimeout(() => this.generateBeep(784, 0.15, 'sine'), 200);
    },
    
    // Play error sound
    playError() {
        if (this.isMuted) return;
        
        // Descending harsh notes
        setTimeout(() => this.generateBeep(400, 0.15, 'sawtooth'), 0);
        setTimeout(() => this.generateBeep(300, 0.2, 'sawtooth'), 150);
    },
    
    // Play alert sound
    playAlert() {
        if (this.isMuted) return;
        
        // Urgent beeping
        for (let i = 0; i < 4; i++) {
            setTimeout(() => this.generateBeep(880, 0.1, 'square'), i * 200);
            setTimeout(() => this.generateBeep(440, 0.1, 'square'), i * 200 + 100);
        }
    },
    
    // Play hack sound
    playHack() {
        if (this.isMuted) return;
        
        // Digital noise sequence
        for (let i = 0; i < 8; i++) {
            setTimeout(() => {
                this.generateBeep(200 + Math.random() * 1000, 0.05, 'square');
            }, i * 50);
        }
    },
    
    // Play money sound (cha-ching!)
    playMoney() {
        if (this.isMuted) return;
        
        setTimeout(() => this.generateBeep(1047, 0.1, 'sine'), 0);
        setTimeout(() => this.generateBeep(1319, 0.1, 'sine'), 80);
        setTimeout(() => this.generateBeep(1568, 0.15, 'sine'), 160);
        setTimeout(() => this.generateBeep(2093, 0.2, 'sine'), 240);
    },
    
    // Play arrest siren
    playArrestSiren() {
        if (this.isMuted) return;
        
        let freq = 400;
        let rising = true;
        
        const sirenInterval = setInterval(() => {
            this.generateBeep(freq, 0.05, 'sawtooth');
            
            if (rising) {
                freq += 50;
                if (freq >= 800) rising = false;
            } else {
                freq -= 50;
                if (freq <= 400) rising = true;
            }
        }, 50);
        
        setTimeout(() => clearInterval(sirenInterval), 2000);
    },
    
    // Background music using oscillators
    startBGMusic() {
        if (this.isMuted || !this.audioContext) return;
        if (this.bgMusicInterval) return;
        
        // Simple ambient drone
        const notes = [130.81, 164.81, 196.00, 220.00]; // C3, E3, G3, A3
        let noteIndex = 0;
        
        this.bgMusicInterval = setInterval(() => {
            if (this.isMuted) return;
            
            const oscillator = this.audioContext.createOscillator();
            const gainNode = this.audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(this.audioContext.destination);
            
            oscillator.frequency.value = notes[noteIndex];
            oscillator.type = 'sine';
            
            gainNode.gain.setValueAtTime(0.05, this.audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, this.audioContext.currentTime + 2);
            
            oscillator.start(this.audioContext.currentTime);
            oscillator.stop(this.audioContext.currentTime + 2);
            
            noteIndex = (noteIndex + 1) % notes.length;
        }, 2000);
        
        // Occasional random computer sounds
        this.bgSoundInterval = setInterval(() => {
            if (this.isMuted || Math.random() > 0.3) return;
            this.generateBeep(1000 + Math.random() * 2000, 0.02, 'square');
        }, 3000);
    },
    
    stopBGMusic() {
        if (this.bgMusicInterval) {
            clearInterval(this.bgMusicInterval);
            this.bgMusicInterval = null;
        }
        if (this.bgSoundInterval) {
            clearInterval(this.bgSoundInterval);
            this.bgSoundInterval = null;
        }
    },
    
    // Toggle mute
    toggleMute() {
        this.isMuted = !this.isMuted;
        if (this.isMuted) {
            this.stopBGMusic();
        }
        return this.isMuted;
    },
    
    // Resume audio context (needed for browsers that suspend it)
    resume() {
        if (this.audioContext && this.audioContext.state === 'suspended') {
            this.audioContext.resume();
        }
    }
};

// Initialize on first user interaction (required by browsers)
document.addEventListener('click', () => {
    if (!AudioManager.audioContext) {
        AudioManager.init();
    }
    AudioManager.resume();
}, { once: true });

document.addEventListener('keydown', () => {
    if (!AudioManager.audioContext) {
        AudioManager.init();
    }
    AudioManager.resume();
}, { once: true });

// Export for use in main.js
window.AudioManager = AudioManager;
